/**************************************************************
 * 
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * 
 *************************************************************/



#ifndef SFX2_TEMPLATEFOLDERCACHE_HXX
#define SFX2_TEMPLATEFOLDERCACHE_HXX

#include "svtools/svtdllapi.h"
#include <sal/types.h>

//.........................................................................
namespace svt
{
//.........................................................................

	//=====================================================================
	//= TemplateFolderCache
	//=====================================================================
	class TemplateFolderCacheImpl;
	/** allows to cache the state of the template directories of OOo
		<p>Usually, this class is used together with an instance of a the
		<service scope="com.sun.star.frame">DocumentTemplates</service> service. It allows to scan the template folders
		of the Office, and updates the configuration so that it reflects the most recent state of the folders.<br/>
		As this is an expensive, the TemplateFolderCache has been introduced. It caches the state of the template
		folders, and allows to determine if the DocumentTemplates service needs to be invoked to do the (much more expensive)
		update.</p>
	@example C++
	<listing>
		TemplateFolderCache aTemplateFolders;
		if ( aTemplateFolders.needsUpdate() )
		{
			// store the current state
			aCache.storeState();

			// create the DocumentTemplates instance
			Reference< XDocumentTemplates > xTemplates = ...;

			// update the templates configuration
			xTemplates->update();
		}

		// do anything which relies on a up-to-date template configuration
	</listing>
	*/
	class SVT_DLLPUBLIC TemplateFolderCache
	{
	private:
		TemplateFolderCacheImpl*		m_pImpl;

	public:
		/** ctor.
		@param _bAutoStoreState
			Set this to <TRUE/> if you want the instance to automatically store the state of the template folders upon
			destruction.<br/>
			If set to <FALSE/>, you would epplicitly need to call <method>storeState</method> to do this.<br/>
			If the current state is not known (e.g. because you did not call needsUpdate, which retrieves it),
			it is not retrieved in the dtor, regardless of the <arg>_bAutoStoreState</arg> flag.
		*/
		TemplateFolderCache( sal_Bool _bAutoStoreState = sal_False );
		~TemplateFolderCache( );

		/** determines whether or not the template configuration needs to be updated
		@param _bForceCheck
			set this to <TRUE/> if you want the object to rescan the template folders in every case. The default (<FALSE/>)
			means that once the information has been retrieved in a first call, every second call returns the same result
			as the first one, even if in the meantime the template folders changed.
		@return
			<TRUE/> if the template configuration needs to be updated
		*/
		sal_Bool	needsUpdate( sal_Bool _bForceCheck = sal_False );

		/** stores the current state of the template folders in the cache
		@param _bForceRetrieval
			if set to <TRUE/>, the current state of the template folders is retrieved again, even if it is already known.
			Usually, you set this to <FALSE/>: After calling <method>needsUpdate</method>, the state is know and does not
			need to be read again.
		*/
		void		storeState( sal_Bool _bForceRetrieval = sal_False );
	};

//.........................................................................
}	// namespace svt
//.........................................................................

#endif // SFX2_TEMPLATEFOLDERCACHE_HXX
