/**************************************************************
 * 
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * 
 *************************************************************/



// MARKER(update_precomp.py): autogen include statement, do not remove
#include "precompiled_io.hxx"
#include <rtl/alloc.h>

#include <limits>
#include <string.h>

#include <com/sun/star/uno/Sequence.hxx>

#include <com/sun/star/uno/Exception.hpp>

using namespace ::com::sun::star::uno;

#include "streamhelper.hxx"

namespace io_stm {

void MemFIFO::write( const Sequence< sal_Int8 > &seq )
	throw ( IFIFO_OutOfMemoryException,
			IFIFO_OutOfBoundsException )
{
	try
	{
		writeAt(getSize(), seq );
	}
	catch( IRingBuffer_OutOfMemoryException & )
	{
		throw IFIFO_OutOfMemoryException();
	}
	catch( IRingBuffer_OutOfBoundsException & )
	{
		throw IFIFO_OutOfBoundsException();
	}
}

void MemFIFO::read( Sequence<sal_Int8> &seq , sal_Int32 nBufferLen ) throw (IFIFO_OutOfBoundsException)
{
	try
	{
		readAt(0, seq , nBufferLen);
		forgetFromStart( nBufferLen );
	}
	catch ( IRingBuffer_OutOfBoundsException & )
	{
		throw IFIFO_OutOfBoundsException();
	}
}

void MemFIFO::skip( sal_Int32 nBytesToSkip ) throw ( IFIFO_OutOfBoundsException )
{
	try
	{
		forgetFromStart( nBytesToSkip );
	}
	catch( IRingBuffer_OutOfBoundsException & )
	{
		throw IFIFO_OutOfBoundsException();
	}
}



MemRingBuffer::MemRingBuffer()
{
	m_nBufferLen 			= 0;
	m_p 					= 0;
	m_nStart 				= 0;
	m_nOccupiedBuffer		= 0;
}

MemRingBuffer::~MemRingBuffer()
{
	if( m_p ) {
		rtl_freeMemory( m_p );
	}
}

void MemRingBuffer::resizeBuffer( sal_Int32 nMinSize ) throw( IRingBuffer_OutOfMemoryException)
{
	sal_Int32 nNewLen = 1;

	while( nMinSize > nNewLen ) {
		nNewLen = nNewLen << 1;
	}

	// buffer never shrinks !
	if( nNewLen < m_nBufferLen ) {
		nNewLen = m_nBufferLen;
	}

	if( nNewLen != m_nBufferLen ) {
		m_p = ( sal_Int8 * ) rtl_reallocateMemory( m_p , nNewLen );
		if( !m_p ) {
			throw IRingBuffer_OutOfMemoryException();
		}

		if( m_nStart + m_nOccupiedBuffer > m_nBufferLen ) {
			memmove( &( m_p[m_nStart+(nNewLen-m_nBufferLen)]) , &(m_p[m_nStart]) , m_nBufferLen - m_nStart );
			m_nStart += nNewLen - m_nBufferLen;
		}
		m_nBufferLen = nNewLen;
	}
}


void MemRingBuffer::readAt( sal_Int32 nPos, Sequence<sal_Int8> &seq , sal_Int32 nBytesToRead ) const
														throw(IRingBuffer_OutOfBoundsException)
{
	if( nPos + nBytesToRead > m_nOccupiedBuffer ) {
		throw IRingBuffer_OutOfBoundsException();
	}

	sal_Int32 nStartReadingPos = nPos + m_nStart;
	if( nStartReadingPos >= m_nBufferLen ) {
		nStartReadingPos -= m_nBufferLen;
	}

	seq.realloc( nBytesToRead );

	if( nStartReadingPos + nBytesToRead > m_nBufferLen ) {
		sal_Int32 nDeltaLen = m_nBufferLen - nStartReadingPos;
		memcpy( seq.getArray() , &(m_p[nStartReadingPos]) , nDeltaLen );
		memcpy( &(seq.getArray()[nDeltaLen]), m_p , nBytesToRead - nDeltaLen );
	}
	else {
		memcpy( seq.getArray() , &(m_p[nStartReadingPos]) , nBytesToRead );
	}
}


void MemRingBuffer::writeAt( sal_Int32 nPos, const Sequence<sal_Int8> &seq )
														throw (IRingBuffer_OutOfBoundsException,
																IRingBuffer_OutOfMemoryException )
{
	checkInvariants();
	sal_Int32 nLen = seq.getLength();

	if( nPos < 0 || nPos > std::numeric_limits< sal_Int32 >::max() - nLen )
	{
		throw IRingBuffer_OutOfBoundsException();
	}

	if( nPos + nLen - m_nOccupiedBuffer > 0 ) {
		resizeBuffer( nPos + seq.getLength() );
	}

	sal_Int32 nStartWritingIndex = m_nStart + nPos;
	if( nStartWritingIndex >= m_nBufferLen ) {
		nStartWritingIndex -= m_nBufferLen;
	}

	if( nLen + nStartWritingIndex > m_nBufferLen ) {
		// two area copy
		memcpy( &(m_p[nStartWritingIndex]) , seq.getConstArray(), m_nBufferLen-nStartWritingIndex );
		memcpy( m_p , &( seq.getConstArray()[m_nBufferLen-nStartWritingIndex] ),
										nLen - (m_nBufferLen-nStartWritingIndex) );

	}
	else {
		// one area copy
		memcpy( &( m_p[nStartWritingIndex]), seq.getConstArray() , nLen );
	}
	m_nOccupiedBuffer = Max( nPos + seq.getLength() , m_nOccupiedBuffer );
	checkInvariants();
}


sal_Int32 MemRingBuffer::getSize()  const throw()
{
	return m_nOccupiedBuffer;
}

void MemRingBuffer::forgetFromStart( sal_Int32 nBytesToForget ) throw (IRingBuffer_OutOfBoundsException)
{
	checkInvariants();
	if( nBytesToForget > m_nOccupiedBuffer ) {
		throw IRingBuffer_OutOfBoundsException();
	}
	m_nStart += nBytesToForget;
	if( m_nStart >= m_nBufferLen ) {
		m_nStart = m_nStart - m_nBufferLen;
	}
	m_nOccupiedBuffer -= nBytesToForget;
	checkInvariants();
}


void MemRingBuffer::forgetFromEnd( sal_Int32 nBytesToForget ) throw (IRingBuffer_OutOfBoundsException)
{
	checkInvariants();
	if( nBytesToForget > m_nOccupiedBuffer ) {
		throw IRingBuffer_OutOfBoundsException();
	}
	m_nOccupiedBuffer -= nBytesToForget;
	checkInvariants();
}


void MemRingBuffer::shrink() throw ()
{
	checkInvariants();

	// Up to now, only shrinking of while buffer works.
	// No other shrinking supported up to now.
	if( ! m_nOccupiedBuffer ) {
		if( m_p ) {
			free( m_p );
		}
		m_p = 0;
		m_nBufferLen = 0;
		m_nStart = 0;
	}

	checkInvariants();
}

}
