/**************************************************************
 * 
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * 
 *************************************************************/



#ifndef FRAMEWORK_UNDOMANAGERHELPER_HXX
#define FRAMEWORK_UNDOMANAGERHELPER_HXX

#include "framework/fwedllapi.h"
#include "framework/iguard.hxx"
#include "framework/imutex.hxx"

/** === begin UNO includes === **/
#include <com/sun/star/document/XUndoManager.hpp>
#include <com/sun/star/util/XModifyListener.hpp>
/** === end UNO includes === **/

#include <boost/scoped_ptr.hpp>

namespace svl
{
    class IUndoManager;
}

//......................................................................................................................
namespace framework
{
//......................................................................................................................

	//==================================================================================================================
	//= IMutexGuard
	//==================================================================================================================
    class SAL_NO_VTABLE IMutexGuard : public IGuard
    {
    public:
        /** returns the mutex guarded by the instance.

            Even if the guard currently has not a lock on the mutex, this method must succeed.
        */
        virtual IMutex& getGuardedMutex() = 0;
    };

	//==================================================================================================================
	//= IUndoManagerImplementation
	//==================================================================================================================
    class SAL_NO_VTABLE IUndoManagerImplementation
    {
    public:
        /** returns the IUndoManager interface to the actual Undo stack

            @throws com::sun::star::lang::DisposedException
                when the instance is already disposed, and no IUndoManager can be provided

            @throws com::sun::star::lang::NotInitializedException
                when the instance is not initialized, yet, and no IUndoManager can be provided
        */
        virtual ::svl::IUndoManager&    getImplUndoManager() = 0;

        /** provides access to an UNO interface for the XUndoManager implementation. Used when throwing exceptions.
        */
        virtual ::com::sun::star::uno::Reference< ::com::sun::star::document::XUndoManager >
                                        getThis() = 0;
    };

	//==================================================================================================================
	//= UndoManagerHelper
	//==================================================================================================================
    class UndoManagerHelper_Impl;
    /** helper class for implementing an XUndoManager

        Several of the methods of the class take an IMutexGuard instance. It is assumed that this guard has a lock on
        its mutext at the moment the method is entered. The lock will be released before any notifications to the
        registered XUndoManagerListeners happen.

        The following locking strategy is used for this mutex:
        <ul><li>Any notifications to the registered XUndoManagerListeners are after the guard has been cleared. i.e.
                without the mutex being locked.</p>
            <li>Any calls into the <code>IUndoManager</code> implementation is made without the mutex being locked.
                Note that this implies that the <code>IUndoManager</code> implementation must be thread-safe in itself
                (which is true for the default implementation, SfxUndoManager).</li>
            <li>An exception to the previous item are the <member>IUndoManager::Undo</member> and
                <member>IUndoManager::Redo</member> methods: They're called with the given external mutex being
                locked.</li>
        </ul>

        The reason for the exception for IUndoManager::Undo and IUndoManager::Redo is that those are expected to
        modify the actual document which the UndoManager works for. And as long as our documents are not thread-safe,
        and as long as we do not re-fit <strong>all</strong> existing SfxUndoImplementations to <em>not</em> expect
        the dreaded SolarMutex being locked when they're called, the above behavior is a compromise between "how it should
        be" and "how it can realistically be".
    */
	class FWE_DLLPUBLIC UndoManagerHelper
	{
    public:
        UndoManagerHelper( IUndoManagerImplementation& i_undoManagerImpl );
        ~UndoManagerHelper();

        // life time control
        void disposing();

        // XUndoManager equivalents
        void            enterUndoContext( const ::rtl::OUString& i_title, IMutexGuard& i_instanceLock );
        void            enterHiddenUndoContext( IMutexGuard& i_instanceLock );
        void            leaveUndoContext( IMutexGuard& i_instanceLock );
        void            addUndoAction( const ::com::sun::star::uno::Reference< ::com::sun::star::document::XUndoAction >& i_action, IMutexGuard& i_instanceLock );
        void            undo( IMutexGuard& i_instanceLock );
        void            redo( IMutexGuard& i_instanceLock );
        ::sal_Bool      isUndoPossible() const;
        ::sal_Bool      isRedoPossible() const;
        ::rtl::OUString getCurrentUndoActionTitle() const;
        ::rtl::OUString getCurrentRedoActionTitle() const;
        ::com::sun::star::uno::Sequence< ::rtl::OUString >
                        getAllUndoActionTitles() const;
        ::com::sun::star::uno::Sequence< ::rtl::OUString > 
                        getAllRedoActionTitles() const;
        void            clear( IMutexGuard& i_instanceLock );
        void            clearRedo( IMutexGuard& i_instanceLock );
        void            reset( IMutexGuard& i_instanceLock );
        void            addUndoManagerListener( const ::com::sun::star::uno::Reference< ::com::sun::star::document::XUndoManagerListener >& i_listener );
        void            removeUndoManagerListener( const ::com::sun::star::uno::Reference< ::com::sun::star::document::XUndoManagerListener >& i_listener );

        // XLockable, base of XUndoManager, equivalents
        void            lock();
        void            unlock();
        ::sal_Bool      isLocked();

        // XModifyBroadcaster equivalents
        void            addModifyListener( const ::com::sun::star::uno::Reference< ::com::sun::star::util::XModifyListener >& i_listener );
        void            removeModifyListener( const ::com::sun::star::uno::Reference< ::com::sun::star::util::XModifyListener >& i_listener );

    private:
        ::boost::scoped_ptr< UndoManagerHelper_Impl >   m_pImpl;
	};

//......................................................................................................................
} // namespace framework
//......................................................................................................................

#endif // FRAMEWORK_UNDOMANAGERHELPER_HXX
