/**************************************************************
 * 
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * 
 *************************************************************/



#ifndef INCLUDED_CANVAS_SURFACE_HXX
#define INCLUDED_CANVAS_SURFACE_HXX

#include <basegfx/point/b2ipoint.hxx>
#include <basegfx/point/b2dpoint.hxx>
#include <basegfx/polygon/b2dpolygon.hxx>
#include <basegfx/range/b2drectangle.hxx>
#include <basegfx/vector/b2isize.hxx>
#include <basegfx/matrix/b2dhommatrix.hxx>
#include <canvas/rendering/irendermodule.hxx>
#include <canvas/rendering/icolorbuffer.hxx>
#include <canvas/rendering/isurface.hxx>

#include "surfacerect.hxx"
#include "pagemanager.hxx"

namespace canvas
{
	//////////////////////////////////////////////////////////////////////////////////
	// Surface
	//////////////////////////////////////////////////////////////////////////////////

	/** surfaces denote occupied areas withing pages.

		pages encapsulate the hardware buffers that
		contain image data which can be used for texturing.
		surfaces are areas within those pages.
	*/
	class Surface
	{
    public:

        Surface( const PageManagerSharedPtr&  rPageManager,
                 const IColorBufferSharedPtr& rColorBuffer,
                 const ::basegfx::B2IPoint&   rPos,
                 const ::basegfx::B2ISize&    rSize );
        ~Surface();

        void setColorBufferDirty();

        /** Render the surface content to screen.

            @param fAlpha
            Overall alpha for content

            @param rPos
            Output position
	            
            @param rTransform
            Output transformation (does not affect output position)
        */
        bool draw( double                         fAlpha,
                   const ::basegfx::B2DPoint&     rPos,
                   const ::basegfx::B2DHomMatrix& rTransform );

        /** Render the surface content to screen.

            @param fAlpha
            Overall alpha for content

            @param rPos
            Output position
	            
            @param rArea
            Subset of the surface to render. Coordinate system are
            surface area pixel, given area will be clipped to the
            surface bounds.
	            
            @param rTransform
            Output transformation (does not affect output position)
        */
        bool drawRectangularArea(
            double                         fAlpha,
            const ::basegfx::B2DPoint&     rPos,
            const ::basegfx::B2DRange&     rArea,
            const ::basegfx::B2DHomMatrix& rTransform );

        /** Render the surface content to screen.

            @param fAlpha
            Overall alpha for content

            @param rPos
            Output position
	            
            @param rClipPoly
            Clip polygon for the surface. The clip polygon is also
            subject to the output transformation.
	            
            @param rTransform
            Output transformation (does not affect output position)
        */
        bool drawWithClip( double                           fAlpha,
                           const ::basegfx::B2DPoint&       rPos,
                           const ::basegfx::B2DPolygon&     rClipPoly,
                           const ::basegfx::B2DHomMatrix&   rTransform );

		// private attributes
    private:
        IColorBufferSharedPtr mpColorBuffer;

        // invoking any of the above defined 'draw' methods
        // will forward primitive commands to the rendermodule.
        PageManagerSharedPtr  mpPageManager;

        FragmentSharedPtr     mpFragment;

        // the offset of this surface with regard to the source
        // image. if the source image had to be tiled into multiple
        // surfaces, this offset denotes the relative pixel distance
        // from the source image's upper, left corner
        ::basegfx::B2IPoint   maSourceOffset;

        // the size in pixels of this surface. please note that
        // this size is likely to be smaller than the size of
        // the colorbuffer we're associated with since we
        // maybe represent only a part of it.
        ::basegfx::B2ISize    maSize;

        bool                  mbIsDirty;

		// private methods
    private:
        bool refresh( canvas::IColorBuffer& rBuffer ) const;
        void prepareRendering();

        basegfx::B2DRectangle getUVCoords() const;
        basegfx::B2DRectangle getUVCoords( const ::basegfx::B2IPoint& rPos,
                                           const ::basegfx::B2ISize&  rSize ) const;
	};

	typedef ::boost::shared_ptr< Surface > SurfaceSharedPtr;
}

#endif
