/**************************************************************
 * 
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * 
 *************************************************************/



#ifndef _BGFX_MATRIX_B2DHOMMATRIX_HXX
#define _BGFX_MATRIX_B2DHOMMATRIX_HXX

#include <sal/types.h>
#include <o3tl/cow_wrapper.hxx>

namespace basegfx
{
	class B2DTuple;
    class Impl2DHomMatrix;

	class B2DHomMatrix
	{
    public:
        typedef o3tl::cow_wrapper< Impl2DHomMatrix > ImplType;

	private:
        ImplType                                     mpImpl;

	public:
		B2DHomMatrix();
		B2DHomMatrix(const B2DHomMatrix& rMat);
		~B2DHomMatrix();

		/** constructor to allow setting all needed values for a 3x2 matrix at once. The
			parameter f_0x1 e.g. is the same as using set(0, 1, f)
		 */
		B2DHomMatrix(double f_0x0, double f_0x1, double f_0x2, double f_1x0, double f_1x1, double f_1x2);

		/// unshare this matrix with all internally shared instances
        void makeUnique();

		double get(sal_uInt16 nRow, sal_uInt16 nColumn) const;
		void set(sal_uInt16 nRow, sal_uInt16 nColumn, double fValue);

		/** allow setting all needed values for a 3x2 matrix in one call. The
			parameter f_0x1 e.g. is the same as using set(0, 1, f)
		 */
		void set3x2(double f_0x0, double f_0x1, double f_0x2, double f_1x0, double f_1x1, double f_1x2);

		// test if last line is default to see if last line needs to be
		// involved in calculations
		bool isLastLineDefault() const;

		// Auf Einheitsmatrix zuruecksetzen
		bool isIdentity() const;
		void identity();

		// Invertierung
		bool isInvertible() const;
		bool invert();

		// Normalisierung
		bool isNormalized() const;
		void normalize();

		// Determinante
		double determinant() const;

		// Trace
		double trace() const;

		// Transpose
		void transpose();

		// Rotation
		void rotate(double fRadiant);

		// Translation
		void translate(double fX, double fY);

		// Skalierung
		void scale(double fX, double fY);

		// Shearing-Matrices
		void shearX(double fSx);
		void shearY(double fSy);

		// Addition, Subtraktion
		B2DHomMatrix& operator+=(const B2DHomMatrix& rMat);
		B2DHomMatrix& operator-=(const B2DHomMatrix& rMat);

		// Vergleichsoperatoren
		bool operator==(const B2DHomMatrix& rMat) const;
		bool operator!=(const B2DHomMatrix& rMat) const;

		// Multiplikation, Division mit Konstante
		B2DHomMatrix& operator*=(double fValue);
		B2DHomMatrix& operator/=(double fValue);

		// Matritzenmultiplikation von links auf die lokale
		B2DHomMatrix& operator*=(const B2DHomMatrix& rMat);

		// assignment operator
		B2DHomMatrix& operator=(const B2DHomMatrix& rMat);

		// Help routine to decompose given homogen 3x3 matrix to components. A correction of
		// the components is done to avoid inaccuracies.
		// Zerlegung
		bool decompose(B2DTuple& rScale, B2DTuple& rTranslate, double& rRotate, double& rShearX) const;
	};

	// Addition, Subtraktion
	inline B2DHomMatrix	operator+(const B2DHomMatrix& rMatA, const B2DHomMatrix& rMatB)
	{
		B2DHomMatrix aSum(rMatA);
		aSum += rMatB;
		return aSum;
	}

	inline B2DHomMatrix	operator-(const B2DHomMatrix& rMatA, const B2DHomMatrix& rMatB)
	{
		B2DHomMatrix aDiv(rMatA);
		aDiv -= rMatB;
		return aDiv;
	}

	// Multiplikation, Division mit Konstante
	inline B2DHomMatrix	operator*(const B2DHomMatrix& rMat, double fValue)
	{
		B2DHomMatrix aNew(rMat); 
		aNew *= fValue; 
		return aNew; 
	}

	inline B2DHomMatrix	operator/(const B2DHomMatrix& rMat, double fValue)
	{
		B2DHomMatrix aNew(rMat); 
		aNew *= 1.0 / fValue;
		return aNew; 
	}

	inline B2DHomMatrix	operator*(const B2DHomMatrix& rMatA, const B2DHomMatrix& rMatB)
	{
		B2DHomMatrix aMul(rMatB);
		aMul *= rMatA;
		return aMul;
	}
} // end of namespace basegfx

#endif /* _BGFX_MATRIX_B2DHOMMATRIX_HXX */
